﻿using Microscopic_Traffic_Simulator.ViewModels;
using Microsoft.Win32;
using System;
using System.Windows;

namespace Microscopic_Traffic_Simulator.Views
{
    /// <summary>
    /// Class implements IInteractions interface by message boxes or dialogs.
    /// </summary>
    class GUIInteractions : IInteractions
    {
        /// <summary>
        /// Application instance.
        /// </summary>
        private App app;

        /// <summary>
        /// Constructor for the GUI Interactions.
        /// </summary>
        /// <param name="app">Application instance.</param>
        internal GUIInteractions(App app)
        {
            this.app = app;
        }

        /// <summary>
        /// Reaction to question with answers Yes/No/Cancel by MessageBox.
        /// </summary>
        /// <param name="title">Title of question.</param>
        /// <param name="message">Question itself.</param>
        /// <returns>True/False/null representing Yes/No/null</returns>
        private bool? YesNoCancel(string title, string message)
        {
            MessageBoxResult result = MessageBox.Show(message, title,
                    MessageBoxButton.YesNoCancel, MessageBoxImage.Warning);
            if (result == MessageBoxResult.Yes)
                return true;
            else if (result == MessageBoxResult.No)
                return false;
            else if (result == MessageBoxResult.Cancel)
                return null;
            else
                throw new
                    ApplicationException(app.FindStringResource("UnexpectedErrorMessage"));
        }

        /// <summary>
        /// Returns path for file to be opened by OpenFileDialog.
        /// </summary>
        /// <param name="filter">Filter files possible to select.</param>
        /// <returns>Path for file to be opened.</returns>
        private string GetPathToOpenFile(string filter)
        {
            OpenFileDialog ofd = new OpenFileDialog();
            ofd.Filter = filter;
            bool? result = ofd.ShowDialog();
            if (result == true)
                return ofd.FileName;
            else
                return null;
        }

        /// <summary>
        /// Gets path for file to something be saved to by SaveFileDialog.
        /// </summary>
        /// <param name="filter">Filter files possible to select.</param>
        /// <returns>Path for file to be used to save something.</returns>
        private string GetPathToSaveFile(string filter)
        {
            SaveFileDialog sfd = new SaveFileDialog();
            sfd.Filter = filter;
            bool? result = sfd.ShowDialog();
            if (result == true)
                return sfd.FileName;
            else
                return null;
        }

        /// <summary>
        /// Provide MessageBoxButton to ask question whether to save changes.
        /// </summary>
        /// <returns>True/False/null for Yes/No/Cancel</returns>
        public bool? SaveChangesYesNoCancel()
        {
            return YesNoCancel(app.FindStringResource("MainWindow_Title"),
                app.FindStringResource("SaveChanges"));
        }

        /// <summary>
        /// Provide OpenFileDialog to open the topology file.
        /// </summary>
        /// <returns>Path of file to be opened.</returns>
        public string GetPathToOpenTopologyFile()
        {
            return GetPathToOpenFile(app.FindStringResource("GeometricTopologiesFilter"));
        }

        /// <summary>
        /// Provide SaveFileDialog to get the path to file the topology to be saved to.
        /// </summary>
        /// <returns>Path to file the topology to be saved to.</returns>
        public string GetPathToSaveTopologyFile()
        {
            return GetPathToSaveFile(app.FindStringResource("GeometricTopologiesFilter"));
        }

        /// <summary>
        /// Asking which parameters file to open for loading.
        /// </summary>
        /// <returns>Path to file.</returns>
        public string GetPathToOpenParametersFile()
        {
            return GetPathToOpenFile(app.FindStringResource("ParametersFilter"));
        }

        /// <summary>
        /// Asking which paramters file the data to be saved into.
        /// </summary>
        /// <returns>Path to file.</returns>
        public string GetPathToSaveParametersFile()
        {
            return GetPathToSaveFile(app.FindStringResource("ParametersFilter"));
        }

        /// <summary>
        /// Provide MessageBox to scream error message.
        /// </summary>
        /// <param name="errorMessage">Text of the error message.</param>
        public void ScreamErrorMessage(string errorMessage)
        {
            MessageBox.Show(errorMessage, app.FindStringResource("MainWindow_Title"),
                MessageBoxButton.OK, MessageBoxImage.Error);
        }
    }
}
